import 'dart:io';
import 'package:auto_size_text/auto_size_text.dart';
import 'package:dio/dio.dart';
import 'package:flutter/cupertino.dart';
import 'package:flutter/material.dart';
import 'package:get/get.dart';
import 'package:open_filex/open_filex.dart';
import 'package:path_provider/path_provider.dart';
import 'package:permission_handler/permission_handler.dart';
import '../../../../common/constant/colors.dart';
import '../../../../common/extensions/widget.dart';
import '../../../../features/auth/business_logic/auth_controller.dart';
import '../../../../features/calender/data/models/attachment.dart';
import '../../../../common/widgets/text.dart';

class AttachmentWidget extends StatefulWidget {
  final AttachmentModel attachmentModel;
  const AttachmentWidget({super.key, required this.attachmentModel});
  @override
  State<AttachmentWidget> createState() => _AttachmentWidgetState();
}

class _AttachmentWidgetState extends State<AttachmentWidget> {
  bool isFind = false;
  File? file;
  @override
  void initState() {
    super.initState();
  }

  Future<Directory> _getDownloadDirectory() async {
    if (Platform.isAndroid) {
      final directory = await getExternalStorageDirectory();
      if (directory != null) {
        final downloadDir = Directory('${directory.path}/LBBIS');
        if (!await downloadDir.exists()) {
          await downloadDir.create(recursive: true);
        }
        return downloadDir;
      }
      throw Exception("Failed to get external storage directory.");
    } else if (Platform.isIOS) {
      final directory = await getApplicationDocumentsDirectory();
      final downloadDir = Directory('${directory.path}/LBBIS');
      if (!await downloadDir.exists()) {
        await downloadDir.create(recursive: true);
      }
      return downloadDir;
    } else {
      throw UnsupportedError('Platform not supported');
    }
  }

  Future<void> _checkAndRequestPermission() async {
    if (Platform.isAndroid) {
      if (await Permission.storage.isGranted ||
          await Permission.manageExternalStorage.isGranted) {
        return;
      }
      final status = await Permission.storage.request();
      if (!status.isGranted) {
        final manageStatus = await Permission.manageExternalStorage.request();
        if (!manageStatus.isGranted) {
          throw Exception("Storage permissions denied");
        }
      }
    } else if (Platform.isIOS) {
      return;
    } else {
      throw UnsupportedError('Platform not supported');
    }
  }

  Future<void> findAttachments() async {
    try {
      final directory = await _getDownloadDirectory();
      final filePath = '${directory.path}/${widget.attachmentModel.name}';
      file = File(filePath);
      if (await file!.exists()) {
        setState(() => isFind = true);
        final result = await OpenFilex.open(filePath);
        switch (result.type) {
          case ResultType.done:
            debugPrint("File opened successfully: $filePath");
            break;
          case ResultType.noAppToOpen:
            _showSnackbar(
              'No app found to open this file type: ${widget.attachmentModel.name}',
            );
            break;
          case ResultType.permissionDenied:
            _showSnackbar('Permission denied while opening the file.');
            break;
          case ResultType.error:
            _showSnackbar('Error opening file: ${widget.attachmentModel.name}');
            break;
          default:
            debugPrint("Unexpected result type: ${result.type}");
        }
      } else {
        setState(() => isFind = false);
        _showSnackbar('File not found: ${widget.attachmentModel.name}');
      }
    } catch (e) {
      debugPrint('Error opening attachment: $e');
      _showSnackbar('Error occurred while opening the file');
    }
  }

  Future<void> downloadFile() async {
    try {
      await _checkAndRequestPermission();
      final directory = await _getDownloadDirectory();
      final filePath = '${directory.path}/${widget.attachmentModel.name}';
      file = File(filePath);
      if (await file!.exists()) {
        debugPrint('ℹ️ File already exists at $filePath');
        await findAttachments();
        return;
      }
      final dio = Dio();
      final url =
          "https://lbbis-back.incentive-dev.com/api/attachment/${widget.attachmentModel.id}";
      await dio.download(
        url,
        filePath,
        options: Options(
          headers: {
            "Authorization": "Bearer ${Get.find<AuthController>().token}",
          },
        ),
        onReceiveProgress: (received, total) {
          if (total != -1) {
            debugPrint(
              "Download progress: ${(received / total * 100).toStringAsFixed(0)}%",
            );
          }
        },
      );

      setState(() {
        isFind = true;
      });

      _showSnackbar('✅ Download completed');
      await findAttachments();
    } catch (e) {
      debugPrint('Download error: $e');
      _showSnackbar('❌ Download failed');
    }
  }

  void _showSnackbar(String message) {
    ScaffoldMessenger.of(
      context,
    ).showSnackBar(SnackBar(content: Text(message)));
  }

  @override
  Widget build(BuildContext context) {
    double size = double.tryParse(widget.attachmentModel.size) ?? 0 / 1000;
    return Container(
      margin: const EdgeInsets.symmetric(horizontal: 15, vertical: 25),
      width: Get.width * 0.25,
      height: Get.height * .1,
      decoration: BoxDecoration(
        boxShadow: [
          BoxShadow(
            color: Colors.grey.withOpacity(0.5),
            spreadRadius: 0,
            blurRadius: 9,
          ),
        ],
        color: AppColors.newPrimeColor.withOpacity(0.8),
        borderRadius: BorderRadius.circular(20),
      ),
      child: Stack(
        children: [
          Container(
            width: Get.width * 0.25,
            height: Get.height * .07,
            decoration: const BoxDecoration(
              borderRadius: BorderRadius.only(
                topLeft: Radius.circular(25),
                topRight: Radius.circular(25),
              ),
            ),
            child: const Icon(
              CupertinoIcons.paperclip,
              size: 40,
              color: CupertinoColors.white,
            ),
          ),
          Column(
            mainAxisAlignment: MainAxisAlignment.end,
            children: [
              Container(
                width: Get.width * 0.25,
                height: Get.height * .12,
                decoration: const BoxDecoration(
                  color: Colors.white,
                  borderRadius: BorderRadius.only(
                    topLeft: Radius.circular(15),
                    topRight: Radius.circular(15),
                    bottomLeft: Radius.circular(5),
                    bottomRight: Radius.circular(5),
                  ),
                ),
                child: Column(
                  mainAxisAlignment: MainAxisAlignment.start,
                  crossAxisAlignment: CrossAxisAlignment.start,
                  children: [
                    AutoSizeText(
                      widget.attachmentModel.name.tr,
                      maxLines: 2,
                      minFontSize: 10,
                      maxFontSize: 15,
                    ).paddingSymmetric(horizontal: 10, vertical: 2),
                    Row(
                      children: [
                        RegularText("size : ".tr, size: 12),
                        RegularText("${size.toInt()}Kb".tr, size: 9),
                      ],
                    ).paddingSymmetric(horizontal: Get.width * .02),
                    Container(
                      alignment: Alignment.bottomCenter,
                      child: Container(
                        alignment: Alignment.center,
                        height: Get.height * .03,
                        decoration: BoxDecoration(
                          borderRadius: BorderRadius.circular(50),
                          color: AppColors.appGreen,
                        ),
                        child: Row(
                          mainAxisAlignment: MainAxisAlignment.center,
                          crossAxisAlignment: CrossAxisAlignment.center,
                          children: [
                            isFind
                                ? RegularText(
                                  "Open".tr,
                                  size: 12,
                                  textColor: Colors.white,
                                )
                                : RegularText(
                                  "Download".tr,
                                  size: 12,
                                  textColor: Colors.white,
                                ),
                            isFind
                                ? Icon(Icons.view_headline, color: Colors.white)
                                : Icon(
                                  Icons.download,
                                  color: Colors.white,
                                  size: 20,
                                ),
                          ],
                        ).onTap(() async {
                          if (isFind) {
                            await findAttachments();
                          } else {
                            await downloadFile();
                          }
                        }),
                      ).paddingOnly(bottom: 2),
                    ).expanded(5),
                  ],
                ),
              ),
            ],
          ),
        ],
      ),
    );
  }
}
